from typing import Callable
import numpy as np
import random
    
def virtual(func: Callable):
    def wrapper(*args, **kwargs):
        raise NotImplementedError(
            f"Function {func.__qualname__} is virtual and is not implemented yet.")
    return wrapper

def gaussian(x, mu, sigma):
    return np.exp(-((x - mu) ** 2) / (2 * sigma ** 2)) / (sigma * np.sqrt(2 * np.pi))

def generate_sample(locs: list[tuple[int, int]], size: int, distortion: int=1) -> np.ndarray:
    """
    生成样本
    """
    sample = np.zeros((size, size), dtype=int)
    for loc in locs:
        x, y = loc
        x += random.randint(-distortion, distortion)
        x = max(0, min(size-1, x))
        y += random.randint(-distortion, distortion)
        y = max(0, min(size-1, y))
        sample[y, x] = 1
    return sample

def gbellmf(x, a, b, c):
    """
    Generalized Bell function

    Parameters:
    x (numpy array): Input variable.
    a (float): Controls the width of the function.
    b (float): Controls the shape of the function.
    c (float): Controls the center position of the function.

    Returns:
    numpy array: Output values of the generalized Bell function.
    """
    return 1 / (1 + np.abs((x - c) / a) ** (2 * b))