/*
 * machine.h: derived from the scout machine/tread.h and
 *            ported to the i860.
 */

/*
 * This must not be included by anything but thread.c.  That's
 * why this file is not protected by #ifdefs.
 */

#define traceLog(v,l,a) 0

/*
 * i860 register usage (Paragon OSF1): 
 *
 * Callee saved registers:  r4-r15, f2-f7
 *
 *      r16-r17 = function return value (non-fp)
 *      r3      = frame pointer
 *      r1      = return address
 *      r2      = stack pointer
 *      f8-f15  = return value (fp)
 *      f0-f1   = always 0.0
 */

#define MAGIC_COOKIE		0xFEEDF00D
#define MIN_STACK_SIZE		(1024)	/* enough for a single printf() */
#define MIN_USEFUL_STACK_SIZE	(2*MIN_STACK_SIZE)
#define DEFAULT_STACK_SIZE	(8*MIN_USEFUL_STACK_SIZE)
#define FPU_STATE_SIZE		(32*8)
#define CPU_STATE_SIZE		(56)

/*
 * Switch to stack TO_SP, discarding the old stack pointer and invoke
 * start() (makes use of delay slot).
 */

/* The first intruction in the sequence below is used to make gdb understand
 * it reached the top of a stack. It is not completely clean, since gdb does
 * not behave exactly as when one requests a backtrace of an usual program:
 * it shows a return address 0x0 to a non-named function ("??") as the top
 * of the stack. But it works on making gdb stop at a safe place, so I guess
 * it's OK, at least for now.
 */
#define SWITCH_STACK_AND_START(to_sp,start)				\
{									\
    asm volatile ("  mov r0, r1       # zero return address
		     br _start        # uses delay slot to set new stack
		     mov %0, sp       # reg->reg mov (1 instruction)
		  " : : "r"(to_sp), "i"(start));	                \
}

/*
 * Save necessary state on stack, save new stack pointer to
 * variable SAVED_SP:
 */
#define SAVE_STATE(saved_sp)						\
{									\
    asm volatile							\
      ("addu -56, sp, sp	# allocate space for state on stack
	st.l  r4,  0(sp)	# save callee-saved registers
	st.l  r5,  4(sp)	# r4-r15 and r3
	st.l  r6,  8(sp)
	st.l  r7, 12(sp)
	st.l  r8, 16(sp)
	st.l  r9, 20(sp)
	st.l r10, 24(sp)
	st.l r11, 28(sp)
	st.l r12, 32(sp)
	st.l r13, 36(sp)
	st.l r14, 40(sp)
	st.l r15, 44(sp)
	st.l  fp, 48(sp)	# aka r3
	st.l  r1, 52(sp)	# save return PC
	st.l  sp, %0		# save sp to saved_sp
        " : "m=" (saved_sp) : : "memory");				\
}


/*
 * Load state stored at FROM_SP.
 */
#define LOAD_STATE(from_sp)						\
{									\
    asm volatile							\
      ("ld.l     %0, sp 	# restore sp from from_sp
	ld.l  0(sp), r4		# restore callee-saved registers
	ld.l  4(sp), r5		# r4-r15 and r3
	ld.l  8(sp), r6
	ld.l 12(sp), r7
	ld.l 16(sp), r8
	ld.l 20(sp), r9
	ld.l 24(sp),r10
	ld.l 28(sp),r11
	ld.l 32(sp),r12
	ld.l 36(sp),r13
	ld.l 40(sp),r14
	ld.l 44(sp),r15
	ld.l 48(sp), fp		# aka r3
	ld.l 52(sp), r1		# restore return PC
	addu  56, sp, sp 	# pop saved state
	" : : "m" (from_sp));						\
}

/*
 * In the alpha, instruction "trapb" guaranteed all traps were taken
 * at the right point. In the i860 this is much harder. We will just
 * hope for the time being that no fpu-related traps remain at this point.
 */

#define SAVE_FPU_STATE(fpu_state)					\
{									\
    asm volatile							\
      ("fst.d f2, 0(%0)
        fst.d f4, 8(%0)
	ixfr r16, f2                    # temporary copy of r16
	fst.d f6, 16(%0)
	ld.c fsr, r16   		# copy fsr
	st.l r16, 24(%0)                # save fsr
	fxfr f2, r16                    # restore r16
	" : : "r" (fpu_state) : "memory");				\
}
       
#define LOAD_FPU_STATE(fpu_state)					\
{									\
    asm volatile							\
      ("ixfr r16, f2                    # temporary copy of r16
	ld.l 24(%0), r16                # load fsr
	fld.d 16(%0), f6
	st.c r16, fsr   		# store fsr
	fxfr f2, r16                    # restore r16
	fld.d 8(%0), f4
	fld.d 0(%0), f2
	" : : "r" (fpu_state) : "memory");				\
}

/* this is for debugging purposes only: */
#define GET_SAVED_PC(sp)	((void*)(((u_long*)(sp))[7]))
