/*
 * event.c
 *
 * x-kernel v3.3
 *
 * Copyright (c) 1993,1991,1990,1996  Arizona Board of Regents
 *
 */

 /*
 * Event manager.  This module implements a facility for managing
 * timeouts.  Both, scheduling timeouts via evSchedule() and
 * canceling them via eventCancel() are as efficient as possible.
 * This is achieved via a "timing wheel."  It consists of a number
 * (EVENT_WHEEL_SIZE) of queues.  All events with a firing time
 * that have the same low-order bits (modulo EVENT_WHEEL_SIZE)
 * go into the same queue.  With an appropriate choice of
 * EVENT_WHEEL_SIZE, each queue therefore will be relatively
 * short, guaranteeing efficient insertion into a queue.  Furthermore,
 * with each tick, the clock routine has to inspect a single queue
 * only to find events that are ready to fire.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/timers.h>

#include "event.h"
#include "event_i.h"
#include "xk_thread.h"
#include "trace.h"
#include "xk_debug.h"
#include "x_util.h"

#define EVENT_WHEEL_SIZE    128
#define EVENT_CACHE_SIZE     32
#define THREAD_CACHE_SIZE     8

static u_long		soft_time = 0;	/* softClock's idea of "now" */
static u_long		usec_per_tick;	/* interval between ticks */
static struct Event	wheel[EVENT_WHEEL_SIZE];
static struct ThreadOptions options;
static int		nthreads_cached;
static Thread		thread_cache;
static int		nevents_cached;
static Event		event_cache;

static u_long timeNowInTicks( void );
static void   sigalarmHandler( int );

#ifdef XK_THREAD_TRACE

Map             localEventMap;

#endif


u_long
timeNowInTicks( void )
{
  struct timespec tp;

  if ( getclock( TIMEOFDAY, &tp ) < 0 ) {
    Kabort("timeNowInTicks: gettimer failed. Exiting!");
  }
  return ( tp.tv_sec*1000000 + tp.tv_nsec/1000 ) / usec_per_tick;
}

static void event_start(void* arg);		/* forward declaration */

static void
cache_event (Event e)
{
#ifdef XK_THREAD_TRACE
    mapRemoveKey(localEventMap, &(e));
#endif

    if (nevents_cached < EVENT_CACHE_SIZE) {
	xTrace1(event, TR_DETAILED, "event.cache_event: caching event 0x%p", e);
	++nevents_cached;
	e->next = event_cache;
	event_cache = e;
    } else {
	xTrace1(event, TR_DETAILED, "event.cache_event: freeing event 0x%p", e);
	free(e);
    }
}


static Event
get_event (void)
{
    Event e;

    if (nevents_cached > 0) {
	--nevents_cached;
	e = event_cache;
	event_cache = event_cache->next;
	xTrace1(event, TR_DETAILED,
		"event.get_event: picked up event 0x%p from cache", e);
    } else {
	/* no cached event---have to create a new one */
	e = malloc(sizeof(struct Event));
	xTrace1(event, TR_DETAILED,
		"event.get_event: allocated event 0x%p", e);
    }
    assert(e);
    return e;
}


static void
cache_thread(void)
{
    if (nthreads_cached < THREAD_CACHE_SIZE) {
	xTrace1(event, TR_DETAILED,
		"event.cache_thread: caching thread 0x%p", tSelf);
	++nthreads_cached;
	tSelf->next = thread_cache;
	thread_cache = tSelf;
	threadSuspendWithContinuation();
    } else {
	xTrace1(event, TR_DETAILED,
		"event.cache_thread: stopping thread 0x%p", tSelf);
	threadStop();
    }
}


static void
fire (Event e)
{
    Thread t;

    xTrace1(event, TR_FUNCTIONAL_TRACE, "event.fire(event=0x%p)", e);
    if (nthreads_cached > 0) {
	--nthreads_cached;
	t = thread_cache;
	thread_cache = thread_cache->next;
	threadSetContArg(t, e);
	xTrace1(event, TR_DETAILED,
		"event.fire: picked up thread 0x%p from cache", t);
    } else {
#ifndef XK_THREAD_TRACE
	static int nhandlers = 0;
	char buf[32];

	sprintf(buf, "event-handler-%d", ++nhandlers);
	options.name = buf;
#endif
	options.arg = e;
	t = threadCreate(event_start, &options);
	xTrace1(event, TR_DETAILED, "event.fire: created thread 0x%p", t);
    }
    assert(t);
    e->state = E_SCHEDULED;
    threadWakeup(t);
}


/*
 * Enqueue "e" into a circular doubly linked list after element
 * "after".
 */
static void
enqueue (Event e, Event after)
{
    e->prev = after;
    e->next = after->next;
    after->next = e;
    e->next->prev = e;
}


/*
 * Dequeue "e" from a circular doubly linked list.  It is guaranted
 * that the queue will remain non-empty.
 */

#define dequeue( e )                                                        \
do { e->prev->next = e->next; e->next->prev = e->prev; } while (0)


/*
 * When going from time "now" to "now+1", all events that should fire
 * in the interval (now-1*usec_per_tick,now*usec_per_tick] are fired.
 */
static void
soft_clock (void* arg)
{
    u_long time, new_time;
    Event e, q;

    /* atomically read current time: */
    new_time = timeNowInTicks();
    xTrace1(event, TR_DETAILED,
	    "event.soft_clock: %lu ticks", new_time - soft_time);

    for (time = soft_time; time != new_time; ++time) {
	q = &wheel[time % EVENT_WHEEL_SIZE];
	for (e = q->next; e != q && e->deltat == 0; e = e->next) {
	    dequeue(e);
	    fire(e);
	}
	--e->deltat;
    }
    soft_time = new_time;
    /*
     * Put ourselves to rest.  Notice that it is OK if we get woken up
     * before we suspend.
     */
    assert(tSelf);
    threadSuspendWithContinuation();
}


/*
 * Insert event "newEvent" into "q".  The queue is maintained in a
 * sorted order such that the "deltat" value of events in the queue
 * specifies how many revolutions after the *preceeding* event the
 * event fires.
 */
static void
insert (Event new_event, Event head)
{
    Event curr;

    xTrace2(event, TR_FUNCTIONAL_TRACE,
	    "event.insert(new_event=0x%p,head=0x%p)", new_event, head);
    for (curr = head->next; curr != head; curr = curr->next) {
	if (curr->deltat < new_event->deltat) {
	    /* new_event goes after curr: */
	    new_event->deltat -= curr->deltat;
	} else {
	    /* new_event goes in front of curr: */
	    curr->deltat -= new_event->deltat;
	    enqueue(new_event, curr->prev);
	    return;
	}
    }
    /* new_event goes at the end */
    enqueue(new_event, head->prev);
}


/*
 * Fired events start here:
 */
static void
event_start (void* arg)
{
    Event e = (Event) arg;

    xTrace1(event, TR_FUNCTIONAL_TRACE, "event.start(e=0x%p)", e);
    assert(e->state == E_SCHEDULED);

    #ifdef XK_THREAD_TRACE
      xAssert(mapResolve(localEventMap, &e, 0) == XK_SUCCESS);
    #endif

    if (e->flags & E_CANCELLED_F) {
	xTrace1(event, TR_EVENTS,
		"event.start: not starting cancelled event 0x%p", e);
	cache_event(e);
	cache_thread();
	/* cache_thread will never return */
    }

    xTrace1(event, TR_EVENTS,
	    "event.start: invoking event-handler at 0x%p", e->func);
    e->state = E_RUNNING;
    e->func(e, e->arg);
    e->state = E_FINISHED;

    if (e->flags & E_DETACHED_F) {
	cache_event(e);
    }
    cache_thread();
}


void
evInit (unsigned usec)
{
    int i;
    timer_t timer;
    struct itimerspec interval;
    u_long sec;

    xTrace1(event, TR_FUNCTIONAL_TRACE, "eventInit(interval=%u)", usec);
    usec_per_tick = usec;
    soft_time     = timeNowInTicks();

    if ( sec = usec / 1000000 )
      usec %= 1000000;

#ifdef XK_THREAD_TRACE
    localEventMap = mapCreate(128, sizeof(Event));
#endif

    for (i = 0; i < EVENT_WHEEL_SIZE; ++i) {
	wheel[i].next = &wheel[i];
	wheel[i].prev = &wheel[i];
	wheel[i].flags = 0;
    }

    signal( SIGALRM, sigalarmHandler );
    xTrace0(event, TR_EVENTS, "eventInit instaled sigalarmHandler");
 
    interval.it_interval.tv_sec  = sec;
    interval.it_interval.tv_nsec = usec*1000;
    interval.it_value = interval.it_interval;
    /* Using this would stop the clock after the first time:
    interval.it_value.tv_sec  = 0;
    interval.it_value.tv_nsec = 0;
    */

    timer = mktimer(TIMEOFDAY,DELIVERY_SIGNALS,NULL);
    reltimer(timer,&interval,NULL);

    options = *threadDefaultOptions;
    options.name = "event-handler";
    xTrace0(event, TR_FUNCTIONAL_TRACE, "eventInit: exiting");
}

static void
sigalarmHandler( int signal_number )
{
  struct ThreadOptions sig_h_options;
  Thread               t;

  sig_h_options          = *threadDefaultOptions;
  sig_h_options.name     = "soft-clock";
  sig_h_options.priority = THREAD_PRIO_MAX;
  t = threadCreate(soft_clock, &sig_h_options);
  if (!t) {
      Kabort("sigalarmHandler: failed to create soft-clock thread!");
  }
  xTrace2(event, TR_DETAILED,
	  "sigalarmHandler: created thread `%s' at 0x%p",sig_h_options.name,t);
  threadAsyncWakeup(t);
  signal( SIGALRM, sigalarmHandler );
}

Event
evSchedule (EvFunc func, void* arg, unsigned usec)
{
    Event e;
    int nticks;

    xTrace3(event, TR_FUNCTIONAL_TRACE,
	    "evSchedule(handler=0x%p,arg=0x%p,usec=%u)",func, arg, usec);
    e = get_event();
    e->func = func;
    e->arg = arg;

#ifdef XK_THREAD_TRACE
  {
      XTime     offset;
      Binding   b;

      offset.sec = usec / 1E6;
      offset.usec = usec % (unsigned)1E6;
       

      xAssert(localEventMap);
      b = mapBind(localEventMap, &e, e);
      xAssert( b != ERR_BIND );
      e->bind = 0;   /* filled in in 'stub' once a native thread is assigned */
      xGetTime(&e->startTime);
      xTrace2(event, TR_DETAILED, "event start time: %d %d",
              e->startTime.sec, e->startTime.usec);
      xAddTime(&e->startTime, e->startTime, offset);
  }
#endif

    if (usec > 0) {
	/* rounded # of ticks to delay: */
	nticks = (2*usec + usec_per_tick) / (2*usec_per_tick);
	if (nticks == 0) {
	    ++nticks;		/* delay at least one tick */
	}
	/*
	 * Notice: soft_clock can be behind the real time due to other
	 *	   threads not relinquishing the processor frequently
	 *	   enough.  We take care of this by adding the
	 *	   difference between the real time (timeNowInTicks) and
	 *	   softClock's time to nticks:
	 */
	nticks += timeNowInTicks() - soft_time;

	/* compute how many clock-wheel revolutions this needs: */
	e->deltat = nticks / EVENT_WHEEL_SIZE;
	e->flags  = 0;
	e->state  = E_PENDING;
	insert(e, &wheel[(soft_time + nticks) % EVENT_WHEEL_SIZE]);
    } else {
	e->flags  = 0;
	e->deltat = 0;
	fire(e);
    }
    xTrace0(event, TR_FUNCTIONAL_TRACE, "evSchedule: exiting");
    return e;
}


void
evDetach (Event e)
{
    xTrace1(event, TR_FUNCTIONAL_TRACE, "evDetach(e=0x%p)", e);
    if (e->state == E_FINISHED) {
	xTrace1(event, TR_EVENTS, "evDetach: caching event 0x%p", e);
	cache_event(e);
    } else {
	xTrace1(event, TR_EVENTS, "evDetach: marks event 0x%p as detached", e);
	e->flags |= E_DETACHED_F;
    }
    xTrace0(event, TR_FUNCTIONAL_TRACE, "evDetach: exiting");
}


EvCancelReturn
evCancel (Event e)
{
    EvCancelReturn res = EVENT_CANCELLED;

    xTrace1(event, TR_FUNCTIONAL_TRACE, "eventCancel(event=0x%p)", e);
    assert(e);

#ifdef XK_DEBUG
  {
    XkReturn    xkr;

    xkr = mapResolve(localEventMap, &e, 0);
    if( xkr != XK_SUCCESS ) {
      Kabort("Attempt to cancel nonexistent event");
    }
  }
#endif

    e->flags |= E_DETACHED_F | E_CANCELLED_F;
    switch (e->state) {
      case E_SCHEDULED:
	/* 
	 * "start" will catch this event before it gets a chance to run:
	 */
	xTrace2(event, TR_EVENTS,
		"eventCancel: cancelled ev 0x%p in SCHEDULED state-flags %#x",
		e, e->flags);
	break;

      case E_BLOCKED:
      case E_RUNNING:
	res = EVENT_RUNNING;
	break;

      case E_FINISHED:
	res = EVENT_FINISHED;
	cache_event(e);
	break;

      case E_PENDING:
	e->next->deltat += e->deltat;
	dequeue(e);
	cache_event(e);
	break;

      default:
	Kabort("eventCancel: bad event state");
    }
    return res;
}


int
evIsCancelled(Event e)
{
    assert(e);
    return e->flags & E_CANCELLED_F;
}

void
evCheckStack( char* str )
{
  fprintf(stderr,"evCheckStack(%s) not implemented for the Paragon\n",str);
}
