/*     
 * $RCSfile: $
 *
 * x-kernel v3.3
 *
 * Copyright (c) 1993,1991,1990,1996  Arizona Board of Regents
 *
 */

/*
 * Threads package. (By David Mosberger)
 */

/* It may be a good idea to add trace messages here... */

#include <assert.h>
#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include "xk_debug.h"
#include "xk_fifo.h"
#include "xk_queue.h"
#include "xk_thread.h"
#include "x_util.h"

#include "xk_machine.h"

#define NPRIOS	(THREAD_PRIO_MIN - THREAD_PRIO_MAX + 1)

struct ThreadStack {
    union {
	void *		sp;		/* top of stack */
	ThreadStack	next;		/* next free stack */
    } u;
    void *	stack_limit;
    long	magic;		/* magic number */
    double	fpu_state[FPU_STATE_SIZE / sizeof(double)];
};

static void start (void);		/* forward declaration */
static void retire (void);		/* never returns */
static void sched (void);
static void error (const char *msg);

static struct ThreadOptions def_opts = { "noname", 0, THREAD_PRIO_STD };
const ThreadOptions threadDefaultOptions = &def_opts;

/* The original state is saved in originalState with setjmp and can be
 * reloaded by terminator_task, which is asynchronously fired by the trigger.
 */
jmp_buf 		originalState;

static void             terminatorTrigger(int arg);
static void		terminator(void* arg);
static Thread		terminatorThread;

/*
 * Invariant: the currently running thread is "threadSelf" (it is
 * *not* in any ready_q!).
 */

static struct Thread	dummyThread;

Thread			threadSelf = &dummyThread;
int 			threadInitialized = FALSE;

static Thread		dead_man = &dummyThread;
static Thread		idle_thread = 0;
static u_int		high_prio = THREAD_PRIO_MAX;
static struct Queue	ready_q[NPRIOS];
static struct FIFO	async_q;

static ThreadStack	first_stack = 0;

#define ATTACH_STACK(s)				\
{						\
    (s) = first_stack;				\
    if (s) {			                \
	first_stack = first_stack->u.next;	\
    } else {					\
	(s) = mkstack();			\
    }						\
    (s)->u.sp = (s)->stack_limit;		\
}

#define DETACH_STACK(s)				\
{						\
    (s)->u.next = first_stack;			\
    first_stack = (s);				\
}

#ifdef XK_THREAD_TRACE
#  define STACK_CHECK()							\
{									\
    if (threadFreeStack() <= 0 || tSelf->stack->magic != MAGIC_COOKIE)	\
    {									\
	error("stack overflow!");					\
	retire();							\
    }									\
}
#else
# define STACK_CHECK()
#endif /* XK_THREAD_TRACE */


#ifdef XK_THREAD_TRACE

#define MAX_THREADS	256

static Thread	blocked[MAX_THREADS];

#define HASH(a)	(((u_long)(a) >> 3) % MAX_THREADS)

static void
mark_blocked (Thread t)
{
    int i, i0;

    i = i0 = HASH(t);
    while (blocked[i]) {
	i = (i + 1) % MAX_THREADS;
	if (i == i0) {
	    Kabort("thread.mark_blocked: too many threads!");
	}
    }
    blocked[i] = t;
}


static void
mark_unblocked (Thread t)
{
    int i, i0;

    i = i0 = HASH(t);
    while (blocked[i] != t) {
	i = (i + 1) % MAX_THREADS;
	if (i == i0) {
	    Kabort("thread.mark_unblocked: couldn't find thread!");
	}
    }
    blocked[i] = 0;
}


void
threadShowBlocked (void)
{
    Thread t;
    int i;

    printf("%-16s %-16s %-16s name:\n", "tcb address:", "sp:", "pc:");
    for (i = 0; i < MAX_THREADS; ++i) {
	t = blocked[i];
	if (t) {
	    printf("%16p %16p %16p %s\n",
		   t, t->stack,
		   t->running ? GET_SAVED_PC(t->stack->u.sp) : t->func,
		   threadName(t));
	}
    }
}

#endif /* XK_THREAD_TRACE */

static void
terminatorTrigger(int arg)
{
    fprintf(stderr,"Terminator trigger activated, waking thread up!\n");
    threadAsyncWakeup(terminatorThread);
}

static void
terminator(void* arg)
{
    fprintf(stderr,"Terminator thread executing longjmp to threadInit...\n");
    longjmp(originalState,1);
}


/*
 * Print string using minimal amount of stack space:
 */
static void
error (const char *msg)
{
    int ch;

    while ((ch = *msg++) != '\0') {
	putchar(ch);
    }
    putchar('\n');
}


static ThreadStack
mkstack (void)
{
    ThreadStack s;

    /*
     * For now, we have only one size of stacks (i.e., we ignore stack
     * size option).
     */
    s = malloc(DEFAULT_STACK_SIZE);
    if (!s) {
	Kabort("thread.mkstack: out of memory");
    }
    s->stack_limit = (char *) s + DEFAULT_STACK_SIZE;
    s->magic = MAGIC_COOKIE;
    memset(s->fpu_state, 0, FPU_STATE_SIZE);
    return s;
}

/*
 * Instead of an endless loop calling threadYield(), use
 * continuations.  This avoids having to waste a stack for the idle
 * thread.
 */
static void
idler (void* arg)
{
    queueAppend(&ready_q[tSelf->option.priority], tSelf);
    /*
    threadSuspendWithContinuation();
    */
    tSelf->running = FALSE;
    DETACH_STACK(tSelf->stack);
    tSelf->stack = 0;
    threadSelf = &dummyThread;
    sched();
}


/*
 * Switch context from tSelf to t.  This is tricky code.  You
 * should verify the generated assembly code.
 */
static void
cswtch (Thread t)
{
    static Thread old;

    old = threadSelf;
    threadSelf = t;
    if (old != &dummyThread) {
	SAVE_STATE(old->stack->u.sp);
	SAVE_FPU_STATE(old->stack->fpu_state);
    }
    if (tSelf->running) {
	LOAD_STATE(tSelf->stack->u.sp);
	LOAD_FPU_STATE(tSelf->stack->fpu_state);
    } else {
	tSelf->running = TRUE;
	/*
	 * Switch to new stack and invoke start(); the second argument
	 * is just there to tell GCC that start() is really used.
	 */
	SWITCH_STACK_AND_START(tSelf->stack->u.sp, start);
	error("Should never return from SWITCH_STACK_AND_START");
    }
}


static void
sched (void)
{
    Thread next;

    /* now is the time move asynchronously awakened threads: */
    while ((next = fifoRemove(&async_q)) != 0) {
#ifdef XK_THREAD_TRACE
	mark_unblocked(next);
#endif
	assert(next->option.priority <= THREAD_PRIO_MIN);
	queueAppend(&ready_q[next->option.priority], next);
	/* maintain high_prio hint: */
	if (next->option.priority < high_prio) {
	    high_prio = next->option.priority;
	}
    }

    /* dispatch to highest priority thread: */
    while (1) {
	assert(high_prio <= THREAD_PRIO_MIN);
	next = (Thread) queueRemove(&ready_q[high_prio]);
	if (next) {
#ifdef PERFMON
	    /* tSelf maybe 0 at this point, so don't rely on it! */
	    if (next == idle_thread) {
		ev4_disable_counters();
	    } else {
		ev4_enable_counters();
	    }
#endif
	    if (!next->stack) {
		ATTACH_STACK(next->stack);
	    }
	    cswtch(next);
	    if (dead_man != &dummyThread) {
		xTrace2(processswitch, TR_EVENTS,
			"thread.sched: freeing space of dead thread `%s' at %p",
			 threadName(dead_man), dead_man);
		DETACH_STACK(dead_man->stack);
		free(dead_man);
		dead_man = &dummyThread;
	    }
	    return;
	}
	++high_prio;
    }
}


static Thread
create (ThreadFunc func, const ThreadOptions options)
{
    Thread t;
    char * mem;
    int name_len;

    name_len = strlen(options->name) + 1;

    if (options->priority > THREAD_PRIO_MIN) {
	xTrace1(processcreation, TR_ERRORS,
		 "thread.create: illegal thread priority %d specified",
		  options->priority);
	return 0;
    }

    /*
     * Allocate thread descriptor, and name string in one chunk:
     */
    mem = malloc(sizeof(struct Thread) + name_len);
    if (!mem) {
	xTrace0(processcreation, TR_EVENTS,
		 "thread.create: not enough memory for new thread");
	return 0;
    }

    /* initialize memory: */
    t = (Thread) mem;
    t->option		= *options;
    t->option.name	= (char *) t + sizeof(struct Thread);
    t->running		= FALSE;
    t->func		= func;
    t->stack		= 0;
    strcpy(t->option.name, options->name);

#ifdef XK_THREAD_TRACE
    mark_blocked(t);
#endif
    return t;
}


/*
 * Retire the currently executing thread and invoke the scheduler.
 * All memory associated with the current thread is freed.
 */
static void
retire (void)
{
    dead_man = tSelf;	/* let next thread deallocate space */
    threadSelf = &dummyThread;		/* avoid state saving */
    sched();
}


/*
 * Invoke the initial function of thread tSelf.  If it ever
 * returns, we simply retire the thread.
 */
static void
start (void)
{
    if (dead_man != &dummyThread) {
	xTrace2(processcreation, TR_EVENTS,
		 "thread.start: freeing space of dead thread `%s' at %p",
		  threadName(dead_man), dead_man);
	DETACH_STACK(dead_man->stack);
	free(dead_man);
	dead_man = &dummyThread;
    }

    xTrace3(processcreation, TR_EVENTS,
	     "thread.start: starting thread `%s' at %p free stack %ld bytes",
	      threadName(tSelf), tSelf, threadFreeStack());

    (*tSelf->func)(tSelf->option.arg);

    xTrace2(processcreation, TR_EVENTS,
	   "thread.start: thread `%s' (%p) terminating.",
	      threadName(tSelf), tSelf);
    retire();
}


void
threadInit (ThreadFunc init, void* arg)
{
    struct ThreadOptions to;
    int prio;
    Thread t;

    for (prio = 0; prio < NPRIOS; ++prio) {
	queueInit(&ready_q[prio]);
    }
    fifoInit(&async_q);
#ifdef XK_THREAD_TRACE
    memset(blocked, 0, sizeof(blocked));
#endif

    /* create initial thread: */
    to = def_opts;
    to.name = "initializer";
    to.arg  = arg;
    t = create(init, &to);
    if (!t) {
	Kabort("threadInit: failed to create initializer thread");
    }
    threadWakeup(t);

    /* create idle thread: */
    to = def_opts;
    to.name = "idler";
    to.arg  = 0;
    to.priority = THREAD_PRIO_MIN;
    idle_thread = create(idler, &to);
    if (!idle_thread) {
	Kabort("threadInit: failed to create idler thread");
    }
    threadWakeup(idle_thread);
    threadInitialized = TRUE;

    /* create terminator thread: */
    to = def_opts;
    to.name = "terminator";
    to.arg  = 0;
    to.priority = THREAD_PRIO_MAX;
    terminatorThread = create(terminator, &to);
    if (!terminatorThread) {
	Kabort("threadInit: failed to create terminator thread");
    }
    signal(SIGUSR1,terminatorTrigger);

    if ( setjmp(originalState) ) {
	/* Re-instated from within the thread package, should clean things up */
	fprintf(stderr,"threadInit: returned from setjmp.\n");
	return;
    } else {
	/* terminate boot stack and get things going: */
	retire();
    }
}


Thread
threadCreate (ThreadFunc fun, const ThreadOptions options)
{
    /*
    STACK_CHECK();
    */
    return create(fun, options);
}


void
threadStop (void)
{
    xTrace2(processswitch,TR_EVENTS,"threadStop: thread `%s' (%p) terminating.",
	      threadName(tSelf), tSelf);
    retire();
}


void
threadSuspend (void)
{
    STACK_CHECK();
#ifdef XK_THREAD_TRACE
    mark_blocked(tSelf);
#endif    
    sched();
}


void
threadSuspendWithContinuation (void)
{
    STACK_CHECK();
#ifdef XK_THREAD_TRACE
    mark_blocked(tSelf);
#endif    
    tSelf->running = FALSE;
    DETACH_STACK(tSelf->stack);
    tSelf->stack = 0;
    threadSelf = &dummyThread;		/* no need to save any state */
    sched();
}


void
threadWakeup (Thread t)
{
#ifdef XK_THREAD_TRACE
    mark_unblocked(t);
#endif    
    assert(t->option.priority <= THREAD_PRIO_MIN);
    queueAppend(&ready_q[t->option.priority], t);
    if (t->option.priority < high_prio) {
	high_prio = t->option.priority;
    }
}


void
threadAsyncWakeup (Thread t)
{
    /* do not update high_prio here, we don't want to lock access to it! */
    fifoAppend(&async_q, t);
}


void
threadYield (void)
{
    STACK_CHECK();
    queueAppend(&ready_q[tSelf->option.priority], tSelf);
    sched();
}

size_t
threadFreeStack (void)
{
    char sp;
    size_t free_stack;

    if (!tSelf->running) {
	return  DEFAULT_STACK_SIZE;
	/* This is not true, but should work for now. This is to solve
	   the problem of a thread being created by a signal handler
	   called right when tSelf->stack has been detached.
	*/
    }
    free_stack = (&sp - (char *) tSelf->stack) - MIN_STACK_SIZE;
    return free_stack;
}
